function [Bspec,waxis] = bispwin(y,nfft,wind,overlap)
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% MATLAB code
% File: bispwin.m
%
% Bispectrum estimation using the direct (fft-based) approach.
%
% INPUT:
% y       = data vector or time-series
% nfft    = fft length [default = power of two > nsamp]
% wind    = window specification for frequency-domain smoothing
%           if 'wind' is a scalar, it specifieds the length of the side
%           of the square for the Rao-Gabr optimal window  [default = 5]
%           if 'wind' is a vector, a 2D window will be calculated via
%           w2(i,j) = wind(i) * wind(j) * wind(i+j)
%           if 'wind' is a matrix, it specifies the 2-D filter directly
% nsamp   = nfft - samples per segment [default: such that we have 8 segments]
%           if y is a matrix, segsamp is set to the number of rows
% overlap = percentage overlap [default = 50]
%           if y is a matrix, overlap is set to 0.
%
% OUTPUT:
% Bspec   = estimated bispectrum: an (nfft x nfft) array, with origin
%           at the center, and axes pointing down and to the right.
% waxis   = vector of frequencies associated with the rows and columns
%           of Bspec;  sampling frequency is assumed to be 1.
%
% Note: this M-file has been created on the basis of the HOSA toolbox
% --------------------- parameter checks -----------------------------
nsamp = nfft;
[ly, nrecs] = size(y);
if (ly == 1)
  y  = y(:);
  ly = nrecs; nrecs = 1;
end
if (exist('nfft') ~= 1)
  nfft = 128;
end
if (exist('overlap') ~= 1)
  overlap = 0;
end
if (overlap < 0 || overlap > 50)
  overlap = 50;
end
if (nrecs > 1)
    overlap =  0;
end
if (exist('nsamp') ~= 1)
   nsamp = 0;
end
if (nrecs > 1)
  nsamp = ly;
end
if (nrecs == 1 && nsamp <= 0)
   nsamp = fix(ly/(8-7 * overlap/100));
end
if (nfft  < nsamp)
  nfft = 2^nextpow2(nsamp);
end
nadvance = nsamp*(1 - overlap/100);
nrecs=fix((ly-nsamp*overlap/100)/nadvance);
% ------------------- create the Rao-Gabr 2-D window -------------------------
if (exist('wind') ~= 1)  
   wind = 5; 
end

  winsize = wind;
  if (winsize < 0) 
   winsize = 5;                               % Window length L
  end          
  winsize = winsize - rem(winsize,2) + 1;     % make it odd
  if (winsize > 1)
       delta  = 1/(winsize-1);
       theta  = -1/2:delta:1/2;
       op     = ones(winsize,1) * (theta .^2);     % w(m,n)=m^2
       op     = op + op' + theta' * theta;         % m^2 + n^2 + mn
       opwind = ( 1-4*op)*4*sqrt(3)/pi;
       theta  = -(winsize-1)/2:(winsize-1)/2;
       hex    = ones(winsize,1) * theta;       % m
       hex    = abs(hex) + abs(hex') + abs(hex+hex');
       hex    = (hex < winsize);
       opwind = opwind.*hex*delta^2;
    else
      opwind  = 1;
  end
% ---------------- accumulate triple products ----------------------
Bspec    = zeros(nfft,nfft);
mask     = hankel([1:nfft],[nfft,1:nfft-1]);   % Hankel mask (faster)
locseg   = (1:nsamp)';
for krec = 1:nrecs
  xseg   = y(locseg);
  Xf     = fft(xseg, nfft);
  CXf    = conj(Xf);
  Bspec  = Bspec + (Xf * Xf.') .* ...
  reshape(CXf(mask), nfft, nfft);
  locseg = locseg + nadvance;
end
%%%%%%%%%%%%%%%%%%%%%%
Bspec = Bspec/nsamp/nrecs;   
%%%%%%%%%%%%%%%%%%%%%%
% ----------------- frequency-domain smoothing ------------------------
Bspec = flipud(Bspec);
Bspec = fftshift(Bspec);
  if (winsize > 1)
      lby2  = (winsize-1)/2;      
      Bspec = conv2(Bspec,opwind);
      Bspec = Bspec(lby2+1:lby2+nfft,lby2+1:lby2+nfft);
  end
 Bspec = fftshift(Bspec); 
 Bspec = flipud(Bspec);
 for k1=1:nfft
  Bspec(1,k1) = real(Bspec(1,k1));
  Bspec(k1,1) = real(Bspec(k1,1));
 end 


